import React, { useEffect, useState } from 'react';
import axios from 'axios';
import FormBuilder from '../../common/Builder/FormBuilder';

function IntegrationModal({ appId ="", integrationId ="", onClose }) {
    const [showSuccess, setShowSuccess] = useState(false);
    const [appDetails, setAppDetails] = useState(null);
    const [integrationDetails, setIntegrationDetails] = useState(null);
    const baseURL = new URL(axios.defaults.baseURL).origin;

    useEffect(() => {
        if(appId !=""){
        const fetchAppDetails = async () => {

            try {
                const response = await axios.get(`/appflow/${appId}`);

                if (response.status === 200) {
                    setAppDetails(response.data);
                }
            } catch (error) {
                console.error('Error fetching app details:', error);
            }
        };
        fetchAppDetails();
    }

    }, [appId]);

    useEffect(() => {
        if(integrationId) {
            const fetchIntegrationDetails = async () => {
                try {
                    const response = await axios.get(`/apps/integrations/${integrationId}`);
                    if (response.status === 200) {
                        setIntegrationDetails(response.data);
                    }
                } catch (error) {
                    console.error('Error fetching integration details:', error);
                }
            };

            fetchIntegrationDetails();
        }
    }, [integrationId]);

    const handleFinalSubmission = async (formData) => {
        const userId = localStorage.getItem('user_id');
        const accountId = localStorage.getItem('account_id');
        
        const integrationData = {
            ...formData,
            app_id: appId,
            user_id: userId,
            account_id: accountId
        };

        try {
            if(integrationId) {
                const response = await axios.put(`/integrations/view/${integrationId}`, integrationData);
                if (response.status === 200) setShowSuccess(true);
            } else {
                const response = await axios.post('/appflow/subscribe', integrationData);
                if (response.status === 200) setShowSuccess(true);
            }
        } catch (error) {
            console.error('Error submitting the form:', error);
        }
    };
    
    
    const config = {
        formType: 'chat',
        steps: [
            {
                fields: [
                    { name: 'name', type: 'text', placeholder: 'Enter the app integration name', mandatory: true, label:'Name your integration' },
                    { name: 'api_key', type: 'text', placeholder: 'API Key (optional)', mandatory: true, label:'API Key' },
                    { name: 'token', type: 'text', placeholder: 'Token', mandatory: true, label:'Bearer Token' },
                    { name: 'other_params', type: 'textarea', placeholder: 'Other Parameters', mandatory: false, label:'Add any other parameters (Optional)' }
                ],
                defaultValues: integrationDetails || {}
            }
        ]
    };

    return (
        <div className="modal-overlay">
            <div className="modal-content">
                <h3>{appDetails?.name || integrationDetails?.name} - Integration &nbsp;&nbsp;&nbsp; <a href="#" onClick={onClose}> <span className="material-symbols-outlined">close</span></a></h3>
                
                {showSuccess ? (
                    <div>
                        <p>Integration {integrationId ? 'updated' : 'added'} successfully!</p>
                        <button onClick={onClose}>Close</button>
                    </div>
                ) : (
                <FormBuilder 
                    key={integrationId + JSON.stringify(integrationDetails)} 
                    config={config} 
                    onFinalSubmit={handleFinalSubmission} 
                />
                )}
                {appDetails && (
                  <div>
                    <br/>
                    <img 
                        src={`${baseURL}/public/appicons/${appDetails.approot}.jpg`} 
                        alt={appDetails.name} 
                        style={{ width: '200px' }} 
                    />
                  </div>
                )}
            </div>
        </div>
    );
}

export default IntegrationModal;
